function  [mse,grad,hess,fprice,fprices, resid]= LWM_MSE(parm,M1,M2,oprice,K, rf, TTM,weights,objval_scale)
%==========================================================================================
% This function computes the mean squared difference between market futures call option prices 
% and their theoretical prices based on a mixture of LWM densities
%
% INPUT:
%        parm        : 3(M1+M2)-by-1 vector of parameters (weights, futures, sigmas) 
%                      of the LWM densities in a mixture
%                      Parameters for Lognormal densities always come first
%                      in each subvector (weights, futures and sigmas)
%        M1          : number of Lognormal densities in the mixture
%        M2          : number of Weibull densities in the mixture
%        oprice      : N-by-1 vector of call option prices
%        K           : N-by-1 vector of strike prices
%        rf          : scalar of annualized risk free rate, in decimals.
%        TTM         : scalar of time to maturity in years.
%        weights     : N-by-1 vector of corresponding weights. Default equal weights
%        objval_scale:   Multiplicative scale factor applied to objective function. Default = 1e3
%
% OUTPUT: 
%        mse           : mean squared difference
%        grad          : Analytic gradient vector (d MSE/ d theta)
%        hess          : Analytic Hessian metrix (d^2 MSE/ d theta^2)
%        fprice        : fitted option prices
%        fprices       : fitted option prices per mixture
%        resid         : option residuals
%==========================================================================================
% This ver: 2023/05/24
% Authors: Yifan Li (yifan.li@manchester.ac.uk)
%          Ingmar Nolte (i.nolte@lancaster.ac.uk)
%          Manh Pham (m.c.pham@lancaster.ac.uk)
% Reference: Li, Y., Nolte, I., and Pham, M. C. (2023). Parametric Risk-Neutral 
%          Density Estimation via Finite Lognormal-Weibull Mixtures
%========================================================================================== 

if nargin < 8 || isempty(weights)
    weights = ones(size(oprice));
end

if nargin < 9
    objval_scale = 1e3;
end

N=length(oprice); 
%Decomposing the parameter vector
if size(parm,1) < size(parm,2); parm = parm'; end % Use column representation
M=M1+M2;

w_vec = parm(1:M);
f_vec = parm(M+1:2*M);
sigma = parm(2*M+1:end);

%separate BS and WB parameters
f_vec1 = f_vec(1:M1);
sigma1 = sigma(1:M1);

f_vec2 = f_vec(M1+1:end);
sigma2 = sigma(M1+1:end);


%Computing the mixture-wise prices and option Greeks
if M1>0
    [fprice1, delta1, vega1, Gamma1, xi1, eta1] = LN_f(f_vec1', K,rf,TTM,sigma1', true, true);
else
    fprice1=[];delta1=[];vega1=[];Gamma1=[];xi1=[];eta1=[];
end
if M2>0
    [fprice2, delta2, vega2, Gamma2, xi2, eta2] = WB_f(f_vec2', K,rf,TTM,sigma2',true,true);
else
    fprice2=[];delta2=[];vega2=[];Gamma2=[];xi2=[];eta2=[];
end

fprices = [fprice1 fprice2];
delta = [delta1 delta2];
vega= [vega1 vega2];
Gamma = [Gamma1 Gamma2];
xi = [xi1 xi2];
eta = [eta1 eta2];

W = diag(weights);
%Computing fitted price and MSE
fprice=fprices*w_vec;
resid=oprice-fprice;
mse=resid'*W*resid*objval_scale;

Q_grad=[fprices delta.*w_vec' vega.*w_vec'];
grad=-2*Q_grad'*W*resid*objval_scale; % gradient
hess = 2*LWM_Hessian(w_vec,resid,Q_grad,delta,vega,Gamma,eta,xi,weights)*objval_scale;%hessian

end 





